<?php
/*
Plugin Name: Gestione Scaletta Post
Description: Plugin per gestire e monitorare la scaletta dei titoli da pubblicare come post.
Version: 1.0
Author: Renato Cantarella
Plugin URI:  https://www.renatocantarella.it
Author URI:  https://www.renatocantarella.it
Requires PHP: 7.2.5
*/

if (!defined('ABSPATH')) exit;

// Crea le tabelle alla prima attivazione
register_activation_hook(__FILE__, function () {
    global $wpdb;
    $charset = $wpdb->get_charset_collate();

    $table1 = $wpdb->prefix . 'scaletta_post';
    $sql1 = "CREATE TABLE IF NOT EXISTS $table1 (
        id INT NOT NULL AUTO_INCREMENT,
        titolo TEXT NOT NULL,
        autore_id INT DEFAULT NULL,
        depennato TINYINT(1) DEFAULT 0,
        PRIMARY KEY (id)
    ) $charset;";

    $table2 = $wpdb->prefix . 'scaletta_punti';
    $sql2 = "CREATE TABLE IF NOT EXISTS $table2 (
        user_id INT NOT NULL,
        punti INT DEFAULT 0,
        PRIMARY KEY (user_id)
    ) $charset;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql1);
    dbDelta($sql2);
});

// Aggiunge la metabox nella sidebar del post editor
add_action('add_meta_boxes', function () {
    add_meta_box('scaletta_post_box', 'Scaletta Titoli Post', 'scaletta_render_metabox', 'post', 'side', 'default');
});

function scaletta_render_metabox($post) {
    global $wpdb;
    $table = $wpdb->prefix . 'scaletta_post';
    $titoli = $wpdb->get_results("SELECT * FROM $table WHERE depennato = 0");

    if (empty($titoli)) {
        echo '<p>Nessun titolo da mostrare.</p>';
        return;
    }

    echo '<form method="post">';
    echo '<ul style="margin-left: 0; padding-left: 15px; list-style: none;">';
    foreach ($titoli as $titolo) {
        $checked = get_the_title($post->ID) === $titolo->titolo ? 'checked disabled' : '';
        echo '<li><label><input type="checkbox" name="scaletta_check[]" value="' . esc_attr($titolo->id) . '" ' . $checked . '> ' . esc_html($titolo->titolo) . '</label></li>';
    }
    echo '</ul>';
    echo '<input type="hidden" name="scaletta_verifica" value="1">';
    echo '<button class="button" style="margin-top:10px">Depenna selezionati se pubblicati</button>';
    echo '</form>';
}

// Depennamento anche dalla pagina admin
add_action('admin_init', function () {
    if (!current_user_can('manage_options')) return;
    if (!isset($_POST['admin_depenna_check']) || !isset($_POST['admin_depenna_ids'])) return;

    global $wpdb;
    $table = $wpdb->prefix . 'scaletta_post';
    $punti_table = $wpdb->prefix . 'scaletta_punti';

    $ids = array_map('intval', $_POST['admin_depenna_ids']);
    $user_id = get_current_user_id();

    foreach ($ids as $id) {
        $titolo = $wpdb->get_row($wpdb->prepare("SELECT titolo, autore_id, depennato FROM $table WHERE id = %d", $id));
        if (!$titolo || $titolo->depennato == 1) continue;

        if (get_page_by_title($titolo->titolo, OBJECT, 'post')) {
            $wpdb->update($table, [
                'depennato' => 1,
                'autore_id' => $titolo->autore_id ?: $user_id
            ], ['id' => $id]);

            $assegnato_a = $titolo->autore_id ?? 0;
            $wpdb->query($wpdb->prepare(
                "INSERT INTO $punti_table (user_id, punti) VALUES (%d, 1)
                ON DUPLICATE KEY UPDATE punti = punti + 1",
                $assegnato_a
            ));
        }
    }
});

// Gestione del salvataggio lato editor
add_action('save_post', function ($post_id) {
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if (wp_is_post_revision($post_id)) return;
    if (!current_user_can('edit_post', $post_id)) return;
    if (!isset($_POST['scaletta_verifica']) || !isset($_POST['scaletta_check'])) return;

    global $wpdb;
    $table = $wpdb->prefix . 'scaletta_post';
    $punti_table = $wpdb->prefix . 'scaletta_punti';
    $user_id = get_current_user_id();

    $titoli_selezionati = array_map('intval', $_POST['scaletta_check']);

    foreach ($titoli_selezionati as $id) {
        $titolo = $wpdb->get_row($wpdb->prepare("SELECT titolo, autore_id, depennato FROM $table WHERE id = %d", $id));
        if (!$titolo || $titolo->depennato == 1) continue;

        if (get_page_by_title($titolo->titolo, OBJECT, 'post')) {
            $wpdb->update($table, [
                'depennato' => 1,
                'autore_id' => $titolo->autore_id ?: $user_id
            ], ['id' => $id]);

            $assegnato_a = $titolo->autore_id ?? 0;
            $wpdb->query($wpdb->prepare(
                "INSERT INTO $punti_table (user_id, punti) VALUES (%d, 1)
                ON DUPLICATE KEY UPDATE punti = punti + 1",
                $assegnato_a
            ));
        }
    }
});

// Pagina admin per inserire nuovi titoli e gestire lista
add_action('admin_menu', function () {
    add_menu_page('Scaletta Post', 'Scaletta Post', 'manage_options', 'scaletta-post', 'scaletta_render_admin_page');
});

function scaletta_render_admin_page() {
    global $wpdb;
    $table = $wpdb->prefix . 'scaletta_post';
    $punti_table = $wpdb->prefix . 'scaletta_punti';

    if (!empty($_POST['new_titolo'])) {
        $titolo = sanitize_text_field($_POST['new_titolo']);
        $user_id = isset($_POST['autore_id']) ? intval($_POST['autore_id']) : null;

        $esiste_nella_scaletta = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM $table WHERE titolo = %s", $titolo));

        if ($esiste_nella_scaletta) {
            echo '<div class="notice notice-error"><p>Questo titolo è già presente nella scaletta.</p></div>';
        } elseif (get_page_by_title($titolo, OBJECT, 'post')) {
            echo '<div class="notice notice-warning"><p>Esiste già un post pubblicato con questo titolo.</p></div>';
        } else {
            $wpdb->insert($table, ['titolo' => $titolo, 'autore_id' => $user_id]);
            echo '<div class="updated"><p>Titolo aggiunto!</p></div>';
        }
    }

    if (!empty($_POST['delete_id'])) {
        $wpdb->delete($table, ['id' => intval($_POST['delete_id'])]);
        echo '<div class="updated"><p>Titolo eliminato.</p></div>';
    }

    echo '<div class="wrap"><h1>Gestione Scaletta Post</h1>';
    echo '<form method="POST">';
    echo '<input type="text" name="new_titolo" placeholder="Nuovo titolo" required> ';
    echo '<select name="autore_id"><option value="">-- Tutti --</option>';
    foreach (get_users(['fields' => ['ID', 'display_name']]) as $user) {
        echo '<option value="' . $user->ID . '">' . esc_html($user->display_name) . '</option>';
    }
    echo '</select> ';
    echo '<button class="button button-primary">Aggiungi Titolo</button>';
    echo '</form><hr>';

    $titoli = $wpdb->get_results("SELECT * FROM $table ORDER BY depennato ASC, id DESC");
    echo '<form method="POST">';
    echo '<ul>';
    foreach ($titoli as $titolo) {
        $status = $titolo->depennato ? '✅ <em>(Depennato)</em>' : '';
        $autore = $titolo->autore_id !== null ? get_userdata($titolo->autore_id)->display_name : 'Tutti';
        echo '<li>';
        echo '<label><input type="checkbox" name="admin_depenna_ids[]" value="' . $titolo->id . '"' . ($titolo->depennato ? ' disabled' : '') . '> ' . esc_html($titolo->titolo) . '</label> ' . $status . ' — Autore: ' . esc_html($autore);
        echo ' <button type="submit" name="delete_id" value="' . $titolo->id . '" class="button-link">🗑️</button>';
        echo '</li>';
    }
    echo '</ul>';
    echo '<input type="hidden" name="admin_depenna_check" value="1">';
    echo '<button class="button">Depenna selezionati se pubblicati</button>';
    echo '</form><hr>';

    echo '<h2>Punti Utente</h2><ul>';
    $punti = $wpdb->get_results("SELECT * FROM $punti_table ORDER BY punti DESC");
    foreach ($punti as $riga) {
        $user = $riga->user_id > 0 ? get_userdata($riga->user_id)->display_name : 'Tutti';
        echo '<li>' . esc_html($user) . ': ' . intval($riga->punti) . ' punti</li>';
    }
    echo '</ul></div>';
}
